#version 400 compatibility

/*
Copyright (C) 2019 RRe36

All Rights Reserved unless otherwise explicitly stated.


By downloading this you have agreed to the license and terms of use.
These can be found inside the included license-file or here: https://github.com/rre36/glsl_kappa/blob/master/LICENSE

Violating these terms may be penalized with actions according to the Digital Millennium Copyright Act (DMCA), the Information Society Directive and/or similar laws depending on your country.
*/

#include "/lib/common.glsl"

/*
stand in for sky pass
*/

/*
const int colortex0Format   = RGBA16F;
const int colortex1Format   = RGBA16;
const int colortex2Format   = RGB16;
const int colortex3Format   = RGBA16F;
const int colortex4Format   = RGBA16F;
const int colortex5Format   = RGBA16F;
const int colortex6Format   = RGB16F;

const vec4 colortex0ClearColor = vec4(0.0, 0.0, 0.0, 1.0);
const vec4 colortex3ClearColor = vec4(0.0, 0.0, 0.0, 0.0);

c0  - 4x16 scene color (full)
c1  - 1x16 encoded normals, 1x16 lightmaps, 2x16 specular texture (gbuffer -> composite2)
c2  - 1x16 matID, 1x16 translucency albedo hue  (gbuffer -> composite2)
c3  - 4x16 translucencies  (water -> composite0), bloom (composite7 -> final)
c4  - temporals (full)
*/

const int noiseTextureResolution = 256;

in vec2 coord;

flat in mat4x3 light_color;
flat in mat3x3 sky_color;

flat in vec3 cloud_lightcol;

uniform sampler2D colortex0;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform sampler2D depthtex2;

uniform int frameCounter;
uniform int worldTime;

uniform float aspectRatio;
uniform float eyeAltitude;
uniform float far, near;
uniform float frameTimeCounter;
uniform float cloud_bsfade;
uniform float cloud_lflip;
uniform float wetness;
uniform float rainStrength;

uniform vec2 viewSize;

uniform vec3 upvec, upvecView;
uniform vec3 sunvec, sunvecView;
uniform vec3 moonvec, moonvecView;

uniform vec3 cloud_lvec, cloud_lvecView;
uniform vec3 cameraPosition;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

float depth_lin(float depth) {
    return (2.0*near) / (far+near-depth * (far-near));
}

float get_mie(float x, float g) {
    float temp  = 1.0 + sqr(g) - 2.0*g*x;
    return (1.0 - sqr(g)) / ((4.0*pi) * temp*(temp*0.5+0.5));
}

vec3 get_sky(vec3 viewvec) {
    vec3 v      = -viewvec;
    vec3 hvt    = normalize(-upvecView+v);
    vec3 hvb    = normalize(upvecView+v);
    vec3 sv     = normalize(sunvecView+v);
    vec3 mv     = normalize(moonvecView+v);

    float hor_t = dot(hvt, v);
    float hor_b = dot(hvb, v);
    float sun   = 1.0-dot(sv, v);
    float hor   = linStep(1.0-max(hor_t, hor_b), 0.0, 0.2958);

    float horizon   = pow6(hor);

    float zenith    = 1.0-linStep(1.0-hor_t, 0.0, 0.2958);
        zenith      = exp(-zenith*1.5) * 0.8 + 0.2;

    float vdots     = dot(viewvec, sunvecView);

    float sunscatter = get_mie(vdots, 0.78)*rcp(pi)*(1.0-daytime.w);

    float sun_hint  = saturate(get_mie(vdots, 0.2)*rcp(0.2)*1.35);
        horizon     = mix(horizon, horizon*sun_hint, sqr(daytime.x + daytime.z)*0.8);

    float ld        = linStep(1.0 - hor_t, 0.25, 0.2958);
        ld          = pow3(ld) * 0.96 * sun_hint;
        //ld         *= sqr(linStep(hor_t, 0.85, 1.0)) * 0.95 + 0.05;

    vec3 ldcol      = mix(sky_color[1], light_color[0] * sky_color[1] * 0.9 + light_color[0] * sunscatter, sstep(sunvec.y, -0.1, 0.01));
        ldcol       = mix(ldcol, sky_color[0] * 0.5, sqrt(linStep(hor_b, 0.76, 1.0)) * 0.8);

    vec3 sky    = sky_color[0] * zenith;
        sky     = mix(sky, ldcol, ld);
        sky     = mix(sky, sky_color[1] * 1.2, horizon);
        sky    += light_color[0] * sunscatter * 1.35;
        //sky     = hor > 0.99 ? vec3(horizon) : vec3( 0.0);

    return sky;
}

vec3 get_sun(vec3 viewvec) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(sunvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-linStep(sun, 0.0055, 0.0059);
        //s    *= 1.0-sstep(sun, 0.004, 0.0059)*0.5;

    return s*light_color[0]*400.0;
}
vec3 get_sun(vec3 viewvec, vec3 albedo) {
    albedo      = mix(albedo, vec3(v3avg(albedo)), daytime.y*0.8);
    vec3 v      = -viewvec;
    vec3 sv     = normalize(sunvecView+v);
    float sun   = dot(sv, v);

    float s     = sqr(1.0-linStep(sun, 0.03, 0.25));

    return albedo*s*light_color[0]*30.0*(1.0 - rainStrength * 0.85);
}

vec3 get_moon(vec3 viewvec, vec3 albedo) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(moonvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-linStep(sun, 0.03, 0.08);

    return albedo*s*light_color[3]*8.0;
}


vec3 noise_2d(vec2 pos) {
    return texture(noisetex, pos).xyz;
}

vec2 rotate_coord(vec2 pos, const float angle) {
    return vec2(cos(angle)*pos.x + sin(angle)*pos.y, 
                cos(angle)*pos.y - sin(angle)*pos.x);
}

vec3 get_stars(vec3 spos, vec3 svec) {
    vec3 plane  = svec/(svec.y+length(svec.xz)*0.66);
    float rot   = worldTime*rcp(2400.0);
    plane.x    += rot*0.6;
    plane.yz    = rotate_coord(plane.yz, (25.0/180.0)*pi);
    vec2 uv1    = floor((plane.xz)*768)/768;
    vec2 uv2    = (plane.xz)*0.04;

    vec3 starcol = vec3(0.5, 0.68, 1.0);
        starcol  = mix(starcol, vec3(1.0, 0.9, 0.8), noise_2d(uv2).x);
        starcol  = normalize(starcol)*(noise_2d(uv2*1.5).x+1.0);

    float star  = 1.0;
        star   *= noise_2d(uv1).x;
        star   *= noise_2d(uv1+0.1).x;
        star   *= noise_2d(uv1+0.26).x;

    star        = max(star-0.25, 0.0);
    star        = saturate(star*4.0);

    return star*starcol*0.3*sqrt(daytime.w);
}

/* ------ ambient occlusion and gi ------ */

#include "/lib/light/ao.glsl"

vec2 sincos(float x) {
    return vec2(sin(x), cos(x));
}

/* ------ cloud reflections ------ */

vec3 project_sphere(vec2 coord) {
    coord  *= vec2(tau, pi);
    vec2 lon = sincos(coord.x) * sin(coord.y);
    return vec3(lon.x, cos(coord.y), lon.y);
}

#ifdef cloud_reflections_enabled

    #include "/lib/frag/bluenoise.glsl"

    #define layer2_pass

    #include "/lib/atmos/clouds.glsl"

    vec4 compute_vc(in vec3 wvec, inout float layer_transmittance) {
        bool visible    = wvec.y>0.0;

        if (visible) {
            const float vc_midalt = vcloud_alt+vcloud_depth*0.5;

            vec3 bs     = wvec*((vcloud_alt-64.0)/wvec.y);
            vec3 ts     = wvec*((vc_highedge-64.0)/wvec.y);

            vec3 spos   = bs;
            vec3 epos   = ts;

            float dither = dither_bluenoise_s();

            const int steps = 16;

            vec3 rstep  = (epos-spos)/steps;
            vec3 rpos   = rstep*dither + spos + vec3(cameraPosition.x, 64.0, cameraPosition.z);
            float rlength = length(rstep);

            vec3 scatter    = vec3(0.0);
            float transmittance = 1.0;
            float fade      = 0.0;
            float fdist     = vcloud_clip+1.0;

            vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloud_lightcol : light_color[3]*0.3;
                sunlight   *= cloud_lflip;
            vec3 skylight   = light_color[1] * (1.0 - sqrt(daytime.w)*0.96);

            float vdotl     = dot(wvec, cloud_lvec);

            float pfade     = saturate(cloud_mie(vdotl, 0.65));

            #ifdef vcloud_storyMode
            const float sigma_a = 0.1;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #else
            const float sigma_a = 0.20;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #endif

            for (int i = 0; i<steps; ++i, rpos += rstep) {
                if (transmittance < 0.05) break;
                if (rpos.y < vcloud_alt || rpos.y > vc_highedge) continue;

                float dist  = distance(rpos, cameraPosition);
                float dfade = saturate(dist/vcloud_clip);
                if ((1.0-dfade)<0.01) continue;

                float density   = vcloud_density(rpos);
                if (density<=0.0) continue;

                if (fdist>vcloud_clip) fdist = dist;
                else fdist = mix(fdist, dist, transmittance);

                fade    = linStep(dfade, 0.75, 0.99);

                float extinction = density * sigma_t;
                float stept     = expf(-extinction*rlength);
                float integral  = (1.0 - stept) / sigma_t;

                vec3 result_s   = vec3(0.0);

                #ifdef vcloud_light
                float directod  = vc_directOD(rpos, 5)*sigma_a;
                #else
                float directod  = (1.0-sqr(linStep(rpos.y, vcloud_alt, vc_highedge)))*vcloud_depth*sigma_a;
                #endif

                float skyod     = sqrt(1.0-linStep(rpos.y, vcloud_alt, vc_highedge)) * vcloud_depth * sigma_a * 0.5;

                float powder    = 1.0 - expf(-density * 25.0);

                #ifdef vcloud_storyMode
                    const float dpowder = 1.0;
                #else
                    float dpowder   = mix(powder, 1.0, pfade);
                #endif

                /*
                for (int j = 0; j<5; ++j) {
                    float n     = float(j);

                    float s_d   = sigma_s * pow(0.42, n);    //scatter derivate
                    float t_d   = sigma_t * pow(0.32, n);    //transmittance/attentuation derivate
                    float phase = cloud_phase(vdotl, pow(0.35, n));  //phase derivate

                    result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
                    result_s.y += expf(-skyod*t_d) * powder * s_d;
                }*/

                float phase = cloud_phase(vdotl, 1.0);

                result_s.x += max(expf(-directod * sigma_t), expf(-directod * sigma_t * 0.2) * 0.75) * phase * dpowder * sigma_s * 1.25;
                result_s.y += max(expf(-skyod * sigma_t), expf(-skyod * sigma_t * 0.2) * 0.75) * powder * sigma_s * 1.25;
                
                scatter    += result_s * integral * transmittance;

                transmittance *= stept;
            }

            fdist   = max(fdist, 0.0);
            transmittance = linStep(transmittance, 0.05, 1.0);

            layer_transmittance *= transmittance;

            scatter.x *= 2.0;
            scatter.y *= 1.3;

            vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

            fade        = saturate(1.0-fade);

            float skyfade = expf(-fdist*cloud_atmos_density);
            
            transmittance = mix(1.0, transmittance, skyfade*fade);

            return vec4(color*skyfade*fade, transmittance);
        } else {
            return vec4(0.0, 0.0, 0.0, 1.0);
        }
    }

    #ifdef vcloud2_enabled
        vec4 compute_vc2(in vec3 wvec, inout float layer_transmittance) {
            bool visible    = eyeAltitude<vcloud2_alt ? wvec.y>0.0 : wvec.y<=0.0;

            if (visible) {
                const float vc_midalt = vcloud2_alt+vcloud2_depth*0.5;

                vec3 bs     = wvec*((vcloud2_alt-64.0)/wvec.y);
                vec3 ts     = wvec*((vc2_highedge-64.0)/wvec.y);

                vec3 spos   = bs;
                vec3 epos   = ts;

                float dither = dither_bluenoise();

                const int steps = 10;

                vec3 rstep  = (epos-spos)/steps;
                vec3 rpos   = rstep*dither + spos + vec3(cameraPosition.x, 64.0, cameraPosition.z);
                float rlength = length(rstep);

                vec3 scatter    = vec3(0.0);
                float transmittance = 1.0;
                float fade      = 0.0;
                float fdist     = vcloud_clip+1.0;

                vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloud_lightcol : light_color[3]*0.3;
                    sunlight   *= cloud_lflip;
                vec3 skylight   = light_color[1] * (1.0 - sqrt(daytime.w)*0.96);

                float vdotl     = dot(wvec, cloud_lvec);

                float pfade     = saturate(cloud_mie(vdotl, 0.65));

                #ifdef vcloud_storyMode
                const float sigma_a = 0.1;         //absorption coeff
                const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
                const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
                #else
                const float sigma_a = 0.20;         //absorption coeff
                const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
                const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
                #endif

                for (int i = 0; i<steps; ++i, rpos += rstep) {
                    if (transmittance < 0.05) break;
                    if (rpos.y < vcloud2_alt || rpos.y > vc2_highedge) continue;

                    float dist  = distance(rpos, cameraPosition);
                    float dfade = saturate(dist/vcloud2_clip);
                    if ((1.0-dfade)<0.01) continue;

                    float density   = vcloud2_density(rpos);
                    if (density<=0.0) continue;

                    if (fdist>vcloud_clip) fdist = dist;
                    else fdist = mix(fdist, dist, transmittance);

                    fade    = linStep(dfade, 0.75, 0.99);

                    float extinction = density * sigma_t;
                    float stept     = expf(-extinction*rlength);
                    float integral  = (1.0 - stept) / sigma_t;

                    vec3 result_s   = vec3(0.0);

                    #ifdef vcloud_light
                    float directod  = vc2_directOD(rpos, 5)*sigma_a;
                    #else
                    float directod  = (1.0-sqr(linStep(rpos.y, vcloud_alt, vc_highedge)))*vcloud_depth*sigma_a;
                    #endif

                    float skyod     = sqrt(1.0-linStep(rpos.y, vcloud_alt, vc_highedge)) * vcloud_depth * sigma_a * 0.5;

                    float powder    = 1.0 - expf(-density * 25.0);

                    #ifdef vcloud_storyMode
                        const float dpowder = 1.0;
                    #else
                        float dpowder   = mix(powder, 1.0, pfade);
                    #endif

                    /*
                    for (int j = 0; j<5; ++j) {
                        float n     = float(j);

                        float s_d   = sigma_s * pow(0.42, n);    //scatter derivate
                        float t_d   = sigma_t * pow(0.32, n);    //transmittance/attentuation derivate
                        float phase = cloud_phase(vdotl, pow(0.35, n));  //phase derivate

                        result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
                        result_s.y += expf(-skyod*t_d) * powder * s_d;
                    }*/

                    float phase = cloud_phase(vdotl, 1.0);

                    result_s.x += max(expf(-directod * sigma_t), expf(-directod * sigma_t * 0.2) * 0.75) * phase * dpowder * sigma_s * 1.25;
                    result_s.y += max(expf(-skyod * sigma_t), expf(-skyod * sigma_t * 0.2) * 0.75) * powder * sigma_s * 1.25;
                    
                    scatter    += result_s * integral * transmittance;

                    transmittance *= stept;
                }

                fdist   = max(fdist, 0.0);
                transmittance = linStep(transmittance, 0.05, 1.0);
                transmittance = mix(1.0, transmittance, layer_transmittance);
                scatter *= layer_transmittance;
                layer_transmittance *= transmittance;

                scatter.x *= 1.6;
                scatter.y *= 1.0;

                vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

                fade        = saturate(1.0-fade);

                float skyfade = expf(-fdist*cloud_atmos_density*0.9);
                
                transmittance = mix(1.0, transmittance, skyfade*fade);

                return vec4(color*skyfade*fade, transmittance);
            } else {
                return vec4(0.0, 0.0, 0.0, 1.0);
            }
        }
    #endif
#endif

#include "/lib/atmos/project.glsl"

void main() {
    vec4 scenecol   = stex(colortex0);
    float scenedepth = stex(depthtex1).x;

    vec3 viewpos    = screen_viewspace(vec3(coord, scenedepth));
    vec3 viewvec    = normalize(viewpos);
    vec3 scenepos   = view_scenespace(viewpos);
    vec3 svec       = normalize(scenepos);

    if (!landMask(scenedepth)) {
        vec3 skycol = get_sky(viewvec);

        vec3 sun    = get_sun(viewvec, scenecol.rgb);
        vec3 moonstars = get_moon(viewvec, scenecol.rgb) + get_stars(scenepos, svec);

        scenecol.rgb = skycol + (sun+moonstars)*sstep(svec.y, -0.04, 0.01);
    }

    vec4 return3    = vec4(0.0);

    #ifdef ambientOcclusion_enabled
        vec2 ao_coord   = (coord)*2.0;

        if (clamp(ao_coord, -0.003, 1.003) == ao_coord) {
            vec2 coord  = ao_coord;
            float scenedepth = stex(depthtex1).x;
            float ao    = calculate_dbao(depthtex1, scenedepth, coord);
            return3.x   = ao;
        }
    #endif


    /* ------ sky reflection ------ */

    vec4 return5    = vec4(0.0, 0.0, 0.0, 1.0);

    #ifdef reflections_enabled
    vec2 skyref_coord = (coord-vec2(0.55))*SKYREF_LOD;

    if (clamp(skyref_coord, -0.003, 1.003) == skyref_coord) {
        vec2 coord  = saturate(skyref_coord);
        vec3 svec   = project_sphere(coord)*vec3(1.0, 1.0, -1.0);
        vec3 vvec   = mat3(gbufferModelView)*svec;

        return5.rgb = get_sky(vvec) * (pow3(linStep(svec.y, -0.1, 0.04)) * 0.9 + 0.1);

        #ifdef cloud_reflections_enabled
            float layer_transmittance = 1.0;
            vec4 vcloud = compute_vc(svec, layer_transmittance);

            #ifdef vcloud2_enabled
            if (layer_transmittance >= 0.01) {
                vec4 vcloud2 = compute_vc2(svec, layer_transmittance);
                return5.rgb = return5.rgb*vcloud2.a + vcloud2.rgb;
            }
            #endif

            return5.rgb = return5.rgb*vcloud.a + vcloud.rgb;
        #endif
    }
    #endif

    /* ------ sky color, used for areal perspective on clouds ----- */

    if(!(coord.y >= exp2(-SKY_RENDER_LOD) || coord.x >= exp2(-SKY_RENDER_LOD))) {
        vec3 sceneDir = unprojectSky(coord);
        vec3 skycol   = get_sky(mat3(gbufferModelView) * sceneDir);
        return5.rgb   = skycol;
    }

    /*DRAWBUFFERS:035*/
    gl_FragData[0]  = makeDrawbuffer(scenecol);
    gl_FragData[1]  = clampDrawbuffer(return3);
    gl_FragData[2]  = clampDrawbuffer(return5);
}